/* ============================================================================
 UPC - EETAC - CSD - https://digsys.upc.edu
 
P10: Design phase #3: FSM using a microcontroller and a LCD peripheral for 
representing messages and the TMR0 to generate the transmission baud rate 
P12: Internal peripherals. TMR0.


From design phase#1:
- 4-bit serial transmitter 
- Programming in FSM style for a target chip PIC18F46K22. 
- Compile with XC8 v3.0 or newer. Use standard C99. 
- Generate compilation output files COFF for Proteus simulations or 
ELF for in-circuit debugging/programming the prototype
- A pulse is generated at EoT pin when the transmission ends

Design phase#2:
- LCD kit is connected to some pins in PORTD. Add to this project the 
high-level LCD library functions.
LCD library reference: https://digsys.upc.edu/csd/soft/soft.html#MPLABX

Design phase#3:
- RB0/INT0 is no longer used. The idea is to replace external INT0 where the 
CLK is connected by TMR0
- Transmission of Data_in(3..0) start when an ST button pulse is detected. 
- TMR0 configuration as timer to generate a Tp = 6.666 ms (150 b/s).
This is the TMR0 unit: https://digsys.upc.edu/csd/units/TMR0/TMR0.html

The first idea to study how the complete TRM0 mechanism works implies using 
TMR0 in 8-bit mode and pre-scaler (N1), TMR0 Counter (N2), and software 
post-scaler (N3) Timing period (Tp) = TB  N1  N2  N3 

External XTAL FOSC = 8 MHz;		 TB = 4/FOSC = 500 ns 
var_TMR0_pre-scaler_TC_N1 = 1/32 ==> "100"; 
TMR0 in 8-bit mode ---> var_TMR0_TC_N2 = 139; 
software variable var_TMR0_post-scaler_TC_N3 = 3  (char type);

	Timing period		TP = 500 ns  N1  N2   N3  
			6.672 ms  = 500 ns  32  139  3  ==> 149.88 Hz (0.08% error)

This means TMR0 hardware interrupt flag TMR0IF = 500 ns  32  139  = 2.224 ms
to be used to increment var_TMR_post-scaler variable in the same ISR().

From the data-sheet: 
T0CON: TMR0 CONTROL REGISTER:
TMR0ON: TMR0 On/Off Control bit ==> 1 = Enables TMR0; 0 = Stops TMR0

T0CS: TMR0 Clock Source Select bit ==> 0 => Internal instruction cycle clock
that we name TB = 500 ns time-base [FOSC/4 = 2MHz]
When Source Select bit ==> 1 => TMR0 counts external pulses 

T08BIT: TMR0 8-Bit/16-Bit Control bit ==> 1 = TMR0 is configured as an 8-bit
timer/counter

PSA: TMR0 Pre-scaler Assignment bit ==> 0 = TMR0 prescaler is used. 
TMR0 clock input comes from TMR0 pre-scaler output.

T0PS<2:0>: TMR0 prescaler Select bits ==> 100 => 1/32 => var_TMR_prescaler_N1

Writing an initial value on the TMR2 up counter is like a parallel load (LD),
where Din will establish the number of pulses to count before terminal count 
(overflow) var_TMR0_TC_N2 = 139 
						==> Din = (256 - var_TMR0_TC_N2)  = 117  (139 count)

Finally, to set var_CLK_flag once the TP is reached, the var_TMR0_postscaler
is incremented by software at every TMR0IF interrupt until 
var_TMR0_postscaler = var_TMR0_postscaler_TC_N3 

Prototype adapted to introduce the CSD_PICstick training board
    https://digsys.upc.edu/csd/units/CSD_PICstick/CSD_PICstick.html
	
Target microcontroller: PIC18F46k22 
    https://www.microchip.com/en-us/product/PIC18F46K22 


NOTE:The C code exemplified in this CSD Chapter III is not optimised in any 
sense. The CSD style of programming aims clarity and facilitating the 
comprehension of the basic ideas on microcontrollers (taking advantage of the 
previous content on combinational and sequential systems in Chapters I and II. 

This code is the translation of the plans sketched in paper at:
	https://digsys.upc.edu/csd/P12/P12.html
============================================================================= */

/* On disabling some warnings:
Disable warning 520 (function is never called)! To enable it just remove 
or comment the line below  */ 
#pragma warning disable 520
/* Other warnings can be disabled as compiler options: 
Properties --> XC8 Compiler options --> Additional options 
	-Xparser -Wno-implicit-int-float-conversion  -Wno-implicit-int-conversion
*/

// This general header file includes the device specific headers like
// "pic18f46k22.h". This file contains all the microcontroller declarations.
#include <xc.h>    
#include <stdint.h> //defines values associated with exact-width types
#include "config.h"//This file sets all the microcontroller configuration bits 

#include "lcd.h"	// include LCD library header file
#include <stdio.h>//include 'stdio' header file, required for 'printf' function
/* 	===========================================================================
	Definitions
	======================================================================== */
/* This is the state enumeration. Check the state diagram posted on the web to
figure out which is the meaning of every state */
#define Idle		'A'
#define Start_bit	'B'
#define Data_0		'C'
#define Data_1		'D'
#define Data_2		'E'
#define Data_3		'F'
#define End_T		'G'
#define End_Message	'H'

/* A RAM memory position (a single byte or 'char') "var_current_state" type 
"uint8_t" is used to encode state labels as ASCII capital letters for easy 
debugging. 
A single byte can encode up to 256 states, which is enough for our projects.
Remember that encoding FSM states was done using "sequential", "Gray", 
"one-hot", "Johnson", etc. in our previous Chapter 2 on VHDL and FPGA. */ 


/* 	===========================================================================
	Function prototypes
	======================================================================== */
// The three basic functions to organise the code to emulate a FSM
int main(void);
void init_system (void);
uint8_t state_logic(void); // return 0 when ok
uint8_t output_logic(void);  // return 0 when ok  

//The interrupt service routine ISR() to detect the TMR0 overflow 
// The idea is that at any time the main loop can be interrupted when the TMR0 
// reaches its terminal count. 
void __interrupt() ISR(void);

// The auxiliary functions to read/poll and write port pins and be able to
// organise the software variables, to make the main program hardware
// independent and compatible for many microcontrollers. 
void write_outputs (void);
void read_inputs(void);  

/*	===========================================================================
	Global variables	(6 bytes of RAM memory --> uint8_t type
============================================================================ */
static uint8_t var_Data_in;
static uint8_t var_Serial_out;
static uint8_t var_EoT;

static uint8_t var_ST_flag;	// This flag is going to be set by interrupt 
							//when clicking the ST push button
static uint8_t var_CLK_flag;	// This flag is for timing 
							//the transmission at the baud-rate

/* The state variable to be updated ("asynchronously") every time that 
"state_logic()" is executed. */
static uint8_t var_current_state;	// state variable

// LCD parameters. Our CSD_PICstick will run with an 8 MHz external crystal
uint32_t _XTAL_FREQ = 8000000;  // set clock frequency to 8 MHz

static uint8_t var_LCD_flag;	// This flag is for writing only to the LCD 
								//display when new information is available.

// TMR0 variables:
static uint8_t var_TMR0_prescaler_TC_N1;// pre-scaler terminal count value N1
static uint8_t var_TMR0_TC_N2;				// TMR0 terminal count value N2
static uint8_t var_TMR0_postscaler_TC_N3;// post-scaler terminal count value N3
static uint8_t var_TMR0_postscaler; 	// post-scaler counter


/* 	============================================================================
	Main function. 
  	========================================================================= */
int main(void){
	init_system ();
	while(1) {				// FSM infinite loop
		read_inputs(); 		// poll input switch values
		if (state_logic()) break;   // return 0 when ok
		if (output_logic()) break;
		write_outputs();//The interface with the hardware to convert variables 
						 // into voltages at a given microcontroller pin.
	}
	while (1){
		__nop();	// Assembler instruction meaning "do nothing"
					// It simply wastes 4 OSC cycles  
	/* The system has crashed in an illegal state and needs attention.
This second infinite loop is never reached in normal operation. If by any 
chance the system enters this loop, the only way to scape it is by clicking 
reset MCLR_L button (asynchronous CD_L) or provoking a software reset*/
	} 
}	

/* 	===========================================================================
	Function definitions
	======================================================================== */

//*****************************************************************************
// Initialise the microcontroller system  
//*****************************************************************************
void init_system(void) {
// Initialise starts here:
// All analogue inputs disabled.  
// All unused pins are defined as outputs and reset when initialising
	ANSELA	= 0x00;
	PORTA	= 0x00;			// Reset all Flip-Flops at PORTA
	LATA	= 0x00;
	TRISA	= 0b00000110;	// RA2 is the Data_in(3), RA1 is the Data_in(2)

	ANSELB	= 0x00;
	PORTB	= 0x00;
	LATB	= 0x00;
	TRISB	= 0b00000011;	// PORTB RB1 is the INT1 
						// PORTB RB0 is the INT0 
	ANSELC	= 0x00;
	PORTC	= 0x00;
	LATC	= 0x00;
	TRISC	= 0x00;	// PORTC RC5 is Serial_out, PORTC RC2 is EoT

	ANSELD	= 0x00;
	PORTD	= 0x00;
	LATD	= 0x00;
	TRISD	= 0b11000000;	// RD7 is the Data_in(1), RD6 is the Data_in(0)
							// RD(5..0) are for interfacing the LCD
	ANSELE	= 0x00;
	PORTE	= 0x00;
	LATE	= 0x00;
	TRISE	= 0b00000000;

//Initial situation at "Idle" state:
	var_Serial_out = 0;
	var_EoT = 0;
	var_CLK_flag = 0;
	var_ST_flag = 0;
	var_current_state = Idle;
	var_LCD_flag = 0; 

// Configure INT1  to detect the ST start transmission push-button
	INTEDG1 = 1;	// INT1 edge selection: '1' --> on rising edge 
					//(interrupt after clicking the push-button)
	INT1IF 	= 0;	// Reset INT1 hardware flag					
	INT1IE = 1;		// Enable interrupts from the INT1 pin

/* Disconnected, INT0 code not used, but replaced by TMR0 
// Configure INT0 to set the CLK (baud ratio)  
	INTEDG0 = 1;	// INT0 edge selection: '1' --> on rising edge 
					//(interrupt after CLK edge)
	INT0IF 	= 0;	// Reset INT0 hardware flag						
	INT0IE = 0;		// Disable interrupts from the INT0 pin (CLK))
					// Interrupts only when transmitting 
*/

	IPEN = 0;		// Interrupt priority disabled
	PEIE = 0;		// Disable all the other sources of interrupt
	GIE = 1;		// Global interrupts allowed

// Configure and initialise the LCD peripheral  
	// TRISD  = 0;	// configure all port D pins as outputs
	// ANSELD = 0;	// configure all port D pins as digital pins

// set the hardware connection between the MCU and the LCD
	LCD(&PORTD, 0, 1, 2, 3, 4, 5);

// initialise the LCD module with 16 rows and 2 columns (1602 LCD)
	LCD_Begin(16, 2);
	
// print a list of characters on the display
	LCD_Goto(13, 1);//move cursor to position --> row = 1 and column= 13
	LCD_PutC('L');
	LCD_PutC('C');
	LCD_PutC('D');

	__delay_ms(1000);	// keep the message for 1 second
	LCD_Clear();		// clear the display

	// print a text (string) on the display)
	LCD_Goto(1, 1); // move cursor to position --> column = 1 and row = 1
	LCD_PutS("EETAC- CSD - P11");
	LCD_Goto(1, 2);  
	LCD_PutS(" LCD interface  ");
	__delay_ms(1000);

	LCD_Clear();
	LCD_Goto(1, 1);  
	LCD_PutS("S_TX LCD TMR0...");
	__delay_ms(1000);	
	var_LCD_flag = 1;	// Let's the system print the initial message
}

//*****************************************************************************
// State variable logic routine. The combinational circuit CC1.
// State transitions and loops: arrows in the state diagram 
//*****************************************************************************
uint8_t state_logic(void) {
	uint8_t error = 0;
	
	//if (var_CLK_flag == 1) // Sample the Data_in only when is required 
	//	read_inputs (); 

	switch (var_current_state) {
		case Idle:
			if ( var_ST_flag == 0) { 
				var_current_state = Idle;
		
			} else if ( var_CLK_flag == 1) { 
				var_current_state = Start_bit; // on the rising edge 
				var_ST_flag = 0; 
				var_CLK_flag = 0; 
			}
			else 
		// NOTE: These 'else' loops are not required; added only for clarity
				var_current_state = Idle;
		break;

		case Start_bit:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_0;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Start_bit;
		break;

		case Data_0:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_1;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_0;
		break;

		case Data_1:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_2;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_1;
			break;

		case Data_2:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_3;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_2;
			break;

		case Data_3:
			if (var_CLK_flag == 1){ 
				var_current_state = End_T;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_3;
			break;

		case End_T:
			if (var_CLK_flag == 1){ 
				var_current_state = End_Message;
				var_CLK_flag = 0;	// Clear the flag when used	 
				var_LCD_flag = 1;	// to be able to print the end message
			}
			else
				var_current_state = End_T;
			break;

		case End_Message:
			var_current_state = Idle;
			var_LCD_flag = 1;// to be able to print the Idle message again
			break;

		default:
			error = 1;
	}
	return (error);
}

//*****************************************************************************
// Output logic routine. The combinational circuit CC2. 
// Outputs in brackets in the state diagram. 
//*****************************************************************************
uint8_t output_logic(void) {
	uint8_t error = 0;
	switch (var_current_state) {
		case Idle:
			var_Serial_out = 1;
			var_EoT = 0;	
			if (var_LCD_flag == 1) {
					//write only if there is a new info to print
					// Write the output pins once the output variables are set
				LCD_Goto(1, 1); // Home position
				LCD_PutS("IDLE mode ......");
				LCD_Goto(1, 2);  // Second line
				LCD_PutS("... push ST ....");                     
				var_LCD_flag = 0;  // Clear flag, so that no new rewriting 
			}
			break;

		case Start_bit:
			var_Serial_out = 0; // If you are on the design of the Serial_RX
							// the detection of this falling edge will start 
							// the receiver operations
			var_EoT = 0;	// this start-bit can start capturing the nibble
			break;

		case Data_0:
			var_Serial_out = var_Data_in & 0b00000001;
			var_EoT = 0;
			break;

		case Data_1:	//data right shifting  
			var_Serial_out = (var_Data_in & 0b00000010)>>1;
			var_EoT = 0;
			break;

		case Data_2:
			var_Serial_out = (var_Data_in & 0b00000100)>>2;
			var_EoT = 0;
			break;

		case Data_3:
			var_Serial_out = (var_Data_in & 0b00001000)>>3;
			var_EoT = 0;
			break;

		case End_T:
			var_Serial_out = 1;
			var_EoT = 1;     
			break;

		case End_Message:
			var_Serial_out = 1;//This is marking: Serial_out high when not used
			var_EoT = 0;
// Disable TMR0 interrupts once the transmissions ends. 
			TMR0ON = 0; 		// Stop TMR0
			TMR0IE = 0; 		// Disable TMR0 interrupts  
			var_CLK_flag = 0;	// clear TMR0 software flag
// And here, exceptionally, writing is an obligation to assure EoT duration 
// Check what happens if this function is commented  //write_outputs();
// Why such "bug" in EoT ? 
			write_outputs();// Write the output pins            
			if (var_LCD_flag == 1){//write only if there is a new info to print
					// Write the output pins once the output variables are set
				LCD_Goto(1, 1); // Home position
				LCD_PutS(" Transmission...");
				LCD_Goto(1, 2);  // Second line
				LCD_PutS("......END.......");
				var_LCD_flag = 0;  // Clear flag, so that no new rewriting  
				__delay_ms(500); // To see the end message for half second
			}   
			break;
			
		default:
			error = 1;// If no states have been selected, there is an error
						// and the systems must be stopped 
    }
    return (error);
}

//***************************************************************************
//Interrupt Service Routine for attending external interrupt
//***************************************************************************
void __interrupt() ISR (void){	//"interrupt" is the key word here
	uint8_t var_buf = 0;
	if (INT1IF == 1) {
		var_ST_flag = 1;//Set the variable to move the state diagram and clear 
		// the hardware interrupt flag bit (so that the system can be 
		// interrupted again when another active edge is detected)
		INT1IF = 0; // Clear the hardware flag

/* Initialise the CLK source once start transmission is ordered. Considering 
terminal counts N1, N2 and N3 as variables allows the possibility to configure 
them using knobs in the machine front panel or orders incoming from other 
computers. */
		var_TMR0_prescaler_TC_N1 = 0b00000100; // (1/32)
		var_TMR0_TC_N2 = 138; 
		var_TMR0_postscaler_TC_N3 = 2;
			
		T08BIT = 1; // TMR0 is configured as an 8-bit timer/counter
		T0CS = 0;   // Internal clock is the time-base [FOSC/4 = 2MHz]
		PSA = 0;    // Pre-scaler in use	
			
	// To load TMR0 pre-scaler with TC_N1, preserve T0CON except the T0PS(2..0)
		var_buf = T0CON & 0b11111000; 
		T0CON = var_buf | var_TMR0_prescaler_TC_N1; 
		//T0PS0 = 1;
		//T0PS1 = 1;
		//T0PS2 = 0; // T0PS(2..0) = "100" programs a 1:32 Pre-scale value	
			
		TMR0L = (256 - var_TMR0_TC_N2); // Load TC_N2

		var_TMR0_postscaler = 0;       // Initialise the post-scaler (N3))

		TMR0ON = 1; 		// Start TMR0
		TMR0IE = 1; 		// Enable TMR0 interrupts  
		var_CLK_flag = 0;	// clear TMR0 software flag	
	}
	
	if (TMR0IF == 1) {
/*Reload the TMR0 initial value. This a software load, thus some OSC_CLK pulses
will be wasted, meaning time inaccuracy.
What is a possible way to get more time precision? */
		TMR0L = (256 - var_TMR0_TC_N2);        
		
	// run the post-scaler counter
		if (var_TMR0_postscaler < var_TMR0_postscaler_TC_N3) {
			var_TMR0_postscaler++;
		} 
		else {		// 6.666 ms has elapsed
			var_CLK_flag = 1; // Set the state diagram variable
			var_TMR0_postscaler = 0; // clear post-scaler
		} 
		TMR0IF = 0;
	}
}

/* ============================================================================
 Scan, read, poll or capture inputs and process bits so that we can set our 
 convenient variables for running the FSM.
 Many of the variables are "uint8_t" because we can easily monitor them using
 the watch window for debugging purposes.
============================================================================ */
void read_inputs (void){
	uint8_t var_buf1, var_buf2; 
// Let's read all the port bits and then prepare the variables 
// Like in P9, read the port, mask the bit of interest and set the variable 
	var_buf1 = (PORTA & 0b00000110);
	var_buf1 =  var_buf1 << 1; // This is Data_in(3) and Data_in(2) 

	var_buf2 = (PORTD & 0b11000000);
	var_buf2 =  var_buf2 >> 6; // This is Data_in(1) and Data_in(0) 

	var_Data_in = var_buf2 | var_buf1;

}

/* ============================================================================
Driving the output pins. Let's drive the output ports, converting 
microcontroller variables into electrical signals.
As a rule in CSD, we will write all the port pins in a single instruction
preserving (re-writing) the pins of no interest 
We will write one variable at a time. 
NOTE: LATx and PORTx registers are quite similar but not identical. 
Let us use PORTx for reading data inputs and LATx for writing data outputs
============================================================================ */
void write_outputs(void){
	// These are convenient variables to save partial results while operating
	uint8_t var_buf1, var_buf2, var_buf3;

// Translate into C your planned flowchart of bitwise operations 
// Firstly, let's read the port to save the bits of NO interest while clearing 
// the bits to write, and then move the variables to the pin positions, 
// and write

	var_buf1 = PORTC & 0b11011011; 
	var_buf2 = var_EoT << 2;
	var_buf3 = var_Serial_out << 5; 
	var_buf3 = var_buf3 | var_buf2; // The two variables in place 
	LATC = var_buf3 | var_buf1; // write the port in a single instruction 

// Or instead, in a single C instruction: 
//LATC = (PORTC & 0b11011011) | (var_EoT << 2) | (var_Serial_out << 5); 
}

/* end of file */