/* =============================================================================
 UPC - EETAC - CSD - https://digsys.upc.edu
 
P10: Design phase #1: FSM using a microcontroller
- Simple example of a 4-bit serial asynchronous transmitter 
- Programming in FSM style for a target chip PIC18F46K22. 
- Compile with XC8 v3.0 or newer compiler (standard C99)
- Generate compilation output files COFF for Proteus simulations or
ELF for in-circuit debugging/programming the prototype

This code is the translation of the plans sketched in paper at:
	https://digsys.upc.edu/csd/P10/P10.html

The transmission of Data_in(3..0) start when an ST button pulse is detected at 
the baud-rate configured by the CLK signal. For instance, if CLK = 150 Hz, 
one bit is transmitted every 6.666 s. 
A pulse is generated at EoT pin when the transmission ends. 

NOTE:The C code exemplified in this CSD Chapter III is not optimised in any 
sense. The CSD style of programming aims clarity and facilitating the 
comprehension of the basic ideas on microcontrollers (taking advantage of the 
previous content on combinational and sequential systems in Chapters I and II. 

	
Prototype adapted to introduce the CSD_PICstick training board
	https://digsys.upc.edu/csd/units/CSD_PICstick/CSD_PICstick.html

============================================================================ */

/* On disabling some warnings:
Disable warning 520 (function is never called)! To enable it just remove 
or comment the line below  */ 
#pragma warning disable 520
/* Other warnings can be disabled as compiler options: 
Properties --> XC8 Compiler options --> Additional options 
	-Xparser -Wno-implicit-int-float-conversion -Wno-implicit-int-conversion
*/

// This general header file includes the device specific headers like
// "pic18f46k22.h". This file contains all the microcontroller declarations.
#include <xc.h>
#include <stdint.h> //defines values associated with exact-width types
#include "config.h"//This file sets all the microcontroller configuration bits 
/* 	===========================================================================
		Definitions
	======================================================================== */
/* This is the state enumeration. Check the state diagram posted on the web to
figure out which is the meaning of every state */
#define Idle		'A'
#define Start_bit	'B'
#define Data_0		'C'
#define Data_1		'D'
#define Data_2		'E'
#define Data_3		'F'
#define End_T		'G'

/* A RAM memory position (a single byte or 'char') "var_current_state" type 
"uint8_t" is used to encode state labels as ASCII capital letters for easy 
debugging. 
A single byte can encode up to 256 states, which is enough for our projects.
Remember that encoding FSM states was done using "sequential", "Gray", 
"one-hot", "Johnson", etc. in our previous Chapter 2 on VHDL and FPGA. */ 

/*	===========================================================================
	Function prototypes
	======================================================================== */
// The three basic functions to organise the code to emulate a FSM
int main(void);
void init_system (void); 
uint8_t state_logic(void); // return 0 when ok
uint8_t output_logic(void);  // return 0 when ok  

//The interrupt service routine (ISR) to detect var_CLK_flag (external INT0)
// The idea is that at any time the main loop can be interrupted if a CLK edge 
// is detected. In this case from an external oscillator or push-button. 
void __interrupt() ISR(void);

// The auxiliary functions to read/poll and write port pins and be able to
// organise the software variables, to make the main program hardware
// independent and compatible for many microcontrollers. 
void write_outputs (void);
void read_inputs(void);  


/*  ===========================================================================
	Global variables	(6 bytes of RAM memory --> uint8_t type
============================================================================ */

static uint8_t var_Data_in;
static uint8_t var_Serial_out;
static uint8_t var_EoT;

static uint8_t var_ST_flag;  // This flag is going to be set by interrupt 
						  //when clicking the ST push button
static uint8_t var_CLK_flag;	// This flag is for timing 
							//the transmission at the baud-rate

/* The state variable to be updated ("asynchronously") every time that 
"state_logic()" is executed. */
static uint8_t var_current_state;	// state variable

/* ============================================================================
	Main function. 
	========================================================================= */
int main(void){
	init_system ();
	while(1) {				// FSM infinite loop
		read_inputs(); 		// poll input switch values
		if (state_logic()) break;// return 0 when ok
		if (output_logic()) break;
		write_outputs();//The interface with the hardware to convert variables 
						//into voltages at a given microcontroller pin.
	}
	while (1){
		__nop();	// Assembler instruction meaning "do nothing"
					// It simply wastes 4 OSC cycles  
	/* The system has crashed in an illegal state and needs attention.
This second infinite loop is never reached in normal operation. If by any 
chance the system enters this loop, the only way to scape it is by clicking 
reset MCLR_L button (asynchronous CD_L) or provoking a software reset*/
	} 	
}
/* 	===========================================================================
	Function definitions
	======================================================================== */

//*****************************************************************************
// Initialise the microcontroller system  
//*****************************************************************************
void init_system(void) {
// Initialise starts here:
// All analogue inputs disabled.  
// All unused pins are defined as outputs and reset when initialising
	ANSELA	= 0x00;
	PORTA	= 0x00;		// Reset all Flip-Flops at PORTA
	LATA	= 0x00;
	TRISA	= 0b00000110;	// RA2 is the Data_in(3), RA1 is the Data_in(2)

	ANSELB	= 0x00;
	PORTB	= 0x00;
	LATB	= 0x00;
	TRISB	= 0b00000011;	// PORTB RB1 is the INT1 
							// PORTB RB0 is the INT0 
	ANSELC	= 0x00;
	PORTC	= 0x00;
	LATC	= 0x00;
	TRISC	= 0x00;	// PORTC RC5 is Serial_out, PORTC RC2 is EoT

	ANSELD	= 0x00;
	PORTD	= 0x00;
	LATD	= 0x00;
	TRISD	= 0b11000000; // RD7 is the Data_in(1), RD6 is the Data_in(0)

	ANSELE	= 0x00;
	PORTE	= 0x00;
	LATE	= 0x00;
	TRISE	= 0b00000000;

//Initial situation at "Idle" state:
	var_Serial_out = 0;
	var_EoT = 0;
	var_CLK_flag = 0;
	var_ST_flag = 0;
	var_current_state = Idle;

// Configure INT1
	INTEDG1 = 1;	// INT1 edge selection: '1' --> on rising edge 
					//(interrupt after clicking the push-button)
	INT1IF 	= 0;	// Reset INT1 hardware flag
	INT1IE = 1;		// Enable interrupts from the INT1 pin

// Configure INT0
	INTEDG0 = 1;	// INT0 edge selection: '1' --> on rising edge 
					//(interrupt after CLK edge)
	INT0IF 	= 0;	// Reset INT0 hardware flag
	INT0IE = 0;		// Disable interrupts from the INT0 pin (CLK)

	IPEN = 0;		// Interrupt priority disabled
	PEIE = 0;		// Disable all the other sources of interrupt
	GIE = 1;		// Global interrupts allowed

}

//*****************************************************************************
// State variable logic routine. The combinational circuit CC1.
// State transitions and loops: arrows in the state diagram 
//*****************************************************************************
uint8_t state_logic(void) {
	uint8_t error = 0;
	
	//if (var_CLK_flag == 1) // Sample the Data_in only when is required 
		read_inputs (); 

	switch (var_current_state) {
		case Idle:
			if ( var_ST_flag == 0) { 
				var_current_state = Idle;

			} else if ( var_CLK_flag == 1) { 
				var_current_state = Start_bit; // on the rising edge 
				var_ST_flag = 0; 
				var_CLK_flag = 0; 
			}
			else 
		// NOTE: These 'else' loops are not required; added only for clarity
				var_current_state = Idle;
			break;

		case Start_bit:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_0;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Start_bit;
			break;

		case Data_0:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_1;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_0;
			break;

		case Data_1:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_2;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_1;
			break;

		case Data_2:
			if (var_CLK_flag == 1){ 
				var_current_state = Data_3;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_2;
			break;

		case Data_3:
			if (var_CLK_flag == 1){ 
				var_current_state = End_T;
				var_CLK_flag = 0; // Clear the flag when used
			}
			else
				var_current_state = Data_3;
			break;

		case End_T:
			if (var_CLK_flag == 1){ 
				var_current_state = Idle;
				var_CLK_flag = 0; // Clear the flag when used	
				INT0IE = 0;	//Disable interrupts from CLK when going back to 
						//Idle state after having completed the transmission
			}
			else
				var_current_state = End_T;
			break;

		default:
			error = 1;
	}
	return (error);
}

//*****************************************************************************
// Output logic routine. The combinational circuit CC2. 
// Outputs drawn in red ink brackets in the state diagram. 
//*****************************************************************************
uint8_t output_logic(void) {
	uint8_t error = 0;
	switch (var_current_state) {
		case Idle:
			var_Serial_out = 1;
			var_EoT = 0;
			break;

		case Start_bit:
			var_Serial_out = 0; // If you are on the design of the Serial_RX
							// the detection of this falling edge will start 
							// the receiver operations
			var_EoT = 0;	
			break;

		case Data_0:
			var_Serial_out = var_Data_in & 0b00000001;
			var_EoT = 0;
			break;

		case Data_1:	//data right shifting  
			var_Serial_out = (var_Data_in & 0b00000010)>>1;
			var_EoT = 0;
			break;

		case Data_2:
			var_Serial_out = (var_Data_in & 0b00000100)>>2;
			var_EoT = 0;
			break;

		case Data_3:
			var_Serial_out = (var_Data_in & 0b00001000)>>3;
			var_EoT = 0;
			break;

		case End_T:	// Send the stop bit
			var_Serial_out = 1;
			var_EoT = 1;
			break;
			
		default:
			error = 1;	// If no states have been selected, there is an error
						// and the systems must be stopped 
	}
	return (error);
}

//***************************************************************************
//Interrupt Service Routine for attending external interrupt
// Which is the interrupt source?
//***************************************************************************
void __interrupt() ISR (void){	//"interrupt" is the key word here
	if(INT0IF == 1) {	// Is it an INT0 interrupt ?
// If the code execution reach this point is because a falling edge signal 
// in RB0 has been detected, so let us set the 'software' flag so that the 
// FSM will use it. 
		var_CLK_flag = 1;	//Set the "software flag"	  
		INT0IF = 0;	// clear the "hardware" INT0 interrupt flag
	}
	if (INT1IF == 1) {
		var_ST_flag = 1;//Set the variable to move the state diagram 		
		// Initialise the CLK source to start transmission
		INT0IF = 0; 
		INT0IE = 1; 
		var_CLK_flag = 0;

		INT1IF = 0;		// Clear the hardware INT1 interrupt flag
	}

}

/* ============================================================================
 Scan, read, poll or capture inputs and process bits so that we can set our 
 convenient variables for running the FSM.
 Many of the variables are "uint8_t" because we can easily monitor them using
 the watch window for debugging purposes.
============================================================================ */
void read_inputs (void){
	uint8_t var_buf1, var_buf2; 
// Let's read all the port bits and then prepare the variables 
// Like in P9, read the port, mask the bit of interest and set the variable 
	var_buf1 = (PORTA & 0b00000110);
	var_buf1 =  var_buf1 << 1; // This is Data_in(3) and Data_in(2) 

	var_buf2 = (PORTD & 0b11000000);
	var_buf2 =  var_buf2 >> 6; // This is Data_in(1) and Data_in(0) 

	var_Data_in = var_buf2 | var_buf1;

}

/* ============================================================================
Driving the output pins. Let's drive the output ports, converting 
microcontroller variables into electrical signals.
As a rule in CSD, we will write all the port pins in a single instruction
preserving (re-writing) the pins of no interest 
We will write one variable at a time. 
NOTE: LATx and PORTx registers are quite similar but not identical. 
Let us use PORTx for reading data inputs and LATx for writing data outputs
============================================================================ */
void write_outputs(void){
	// These are convenient variables to save partial results while operating
	uint8_t var_buf1, var_buf2, var_buf3;

// Translate into C your planned flowchart of bitwise operations 
// Firstly, let's read the port to save the bits of NO interest while clearing 
// the bits to write, and then move the variables to the pin positions, 
// and write

	var_buf1 = PORTC & 0b11011011; 
	var_buf2 = var_EoT << 2;
	var_buf3 = var_Serial_out << 5; 
	var_buf3 = var_buf3 | var_buf2; // The two variables in place 
	LATC = var_buf3 | var_buf1; // write the port in a single instruction 

// Or instead, in a single C instruction: 
//LATC = (PORTC & 0b11011011) | (var_EoT << 2) | (var_Serial_out << 5); 
}

/* end of file */