/* ============================================================================
 UPC - EETAC - CSD - http://digsys.upc.edu 

	Tutorials series on learning and adapting MCC Melody. The professional 
	alternative to bare-metal style. 

	CSD P10 tutorial project: 1-digit BCD counter. 
	Design phase #1: Basic FSM
	
	INT0 callback interrupt function connected to the ST/SP push-button
	TMR0 as the var_CLK_flag generator
	Q(3..0) outputs for BCD code
	Terminal count (TC10) output
	Count enable (CE)input switch 
	

	Compile with XC8 v3.0 compiler or never (standard C99)
https://www.microchip.com/en-us/tools-resources/develop/mplab-xc-compilers/xc8
	Generate compilation output files COFF for Proteus simulations or ELF 
	for programming the prototype CSD_PICstick
============================================================================ */
	/* =================================================================
	Prototype adapted to the CSD_PICstick training board
		https://digsys.upc.edu/csd/units/CSD_PICstick/CSD_PICstick.html
	Target microcontroller: PIC18F46K22 
		https://www.microchip.com/en-us/product/PIC18F46K22 

	ST_L push-button at RB0/INT0 
	Switch CE at RC4
	Output LED Q(0) at RB4 
	Output LED Q(1) at RB5
	Output LED Q(2) at RC2 
	Output LED Q(3) at RC3
	TC10 at RC7

---------------------------------------------------------------------
I2C tutorial Step #1:
	8-bit I2C port expander interface MCP23008
	 Adapted from the tutorial 
	"Getting-Started-With-I2C-Using-MSSP-on-PIC18-90003281.pdf"
---------------------------------------------------------------------
Let us initialise the I2C host MSSP2 and connect external 
additional LED using the MCP23008 I2C bus expander 
	 Client address A2 = A1 = A0 = 0 ==> 0x20
 
Project documentation location:
https://digsys.upc.edu/csd/DEE/lab07/PIC18F/Counter_BCD_1digit_LCD/Counter_BCD_1digit_LCD.html

Microchip documentation:
https://onlinedocs.microchip.com/oxy/GUID-5A03F818-B7FC-4062-9792-57D08543B586-en-US-11/index.html
	======================================================================== */

/* On disabling some warnings: Disable warning 520 (function is never called)! 
To enable it just remove or comment the line below  */ 
#pragma warning disable 520
/* Other warnings can be disabled as compiler options: 
Properties --> XC8 Compiler options --> Additional options 
	-Xparser -Wno-implicit-int-float-conversion  -Wno-implicit-int-conversion  
*/
/* 	===========================================================================
	Headers
	======================================================================== */
// This general header "system.h" includes the "pic18f46k22.h" and all the 
// resources used in this application, automatically managed by MCC Melody.
#include "./I2C_s1_prj.X/mcc_generated_files/system/system.h"
#include "./I2C_s1_prj.X/mcc_generated_files/i2c_host/i2c_host_types.h"
#include "./I2C_s1_prj.X/mcc_generated_files/i2c_host/mssp2.h"

/* 	============================================================================
	Definitions 
	========================================================================= */
// The counter control FSM has 4 states to control the push-button and start 
// and stop the counter 
#define Idle		'A'
#define Start_CLK	'B'
#define Counting	'C'
#define Stop_CLK	'D'

// The counter FSM has 10 states to count 1-digit BCD
#define Num_0		'A'
#define Num_1		'B'
#define Num_2		'C'
#define Num_3		'D'
#define Num_4		'E'
#define Num_5		'F'
#define Num_6		'G'
#define Num_7		'H'
#define Num_8		'I'
#define Num_9		'J'

// To run the TMR0 as a timer Tp from the crystal Fosc/4 time base. 

// N1 prescaler is set internally by the MCC to N1 = 1/16
// N3 software post-scaler is set internally by the MCC to N3 = 1
// N2 is the TMR0 counter-register to reload after each overflow interrupt:
#define N2_5Hz 12500 	// (TW = Tp = 500 ns16N21 = 100 ms
#define N2_200Hz 625	// (TW = Tp = 500 ns16N21 = 5 ms
						// TMR0 counter is loaded with (65536 - N2)

// To run and configure the I2C client MCP23008 from the I2C tutorial #1
#define MCP23008_I2C_CLIENT_ADDR	0x20
#define MCP23008_REG_ADDR_IODIR		0x00
#define MCP23008_REG_ADDR_GPIO		0x09
#define MCP23008_PINS_DIG_OUT		0x00	// all pins are outputs
#define MCP23008_PINS_DIG_HIGH		0xFF	// all pins are high
#define MCP23008_DATALENGTH			2


/*	===========================================================================
	Function prototypes (not included in the MCC generated file structure
	======================================================================== */
static void INT0_PB_CallBack(void); // INT0 interrupt handler 
static void TMR0_TP_CallBack(void); //TMR0 interrupt handler
void main(void); // main control flow

// The basic function to organise the code to emulate a FSM
// state logic and output logic in a switch case statement
uint8_t Cntl_FSM(void); // return 0 when OK

uint8_t Counter_FSM(void); // return 0 when OK


// The auxiliary functions to read/poll and write port pins and be able to
// organise the software variables, to make the main program hardware
// independent and compatible for many microcontrollers. 
void write_outputs (void);
void read_inputs(void);  

/* 	===========================================================================
		Global variables and data types
Even if some variables are 'bit' type, we use 'char' (byte) or 
'uint8_t' to make them easy to watch while debugging.
============================================================================ */
static uint8_t var_CE;              // '0' when disabled, '1' when enabled
static uint8_t var_TC10;
static uint8_t var_BCD_code; 

//The flag that will be set in the interrupt service routine to indicate that 
// an start or stop order is detected 
uint8_t var_ST_flag; 

//The flag that will be set in the interrupt service routine to indicate that 
// a new CLK edge to advance the counter is detected 
static uint8_t var_CLK_flag;


/* The state variable to be updated ("asynchronously") every time that 
"FSM()" is executed.*/
static uint8_t var_current_state; 

/* The counter device state variable will be updates synchronously with CLK 
 interrupts generated by TMR0 overflow*/
static uint8_t var_current_state_counter; 

// TODO: Replace TimerX with name of const struct TIMER_INTERFACE, 
// from MCC Generated Files > timer > tmrx.c  ----------
// This is the memory address where the timer callback is saved. 
static const struct TIMER_INTERFACE *Timer = &Counter_CLK; 
	
// I2C 
 const i2c_host_interface_t *I2C = &i2c2_host_host_interface;


/* 	===========================================================================
	main function 
	======================================================================== */
void main(void){
	
	SYSTEM_Initialize();// Automated and maintained by MCC Melody

	// Key elements to install interrupts: A pointer to our interrupt handler, 
	// in a way that we do not need to modify a single MCC generated file. 
	INT0_InterruptHandler = *INT0_PB_CallBack;
	
	// This structure element points to our TMR0 interrupt handler
	Timer->TimeoutCallbackRegister(TMR0_TP_CallBack);
	
	// Initial values 
	read_inputs(); // poll input switch value to set var_CE
	var_TC10 = 0;
	var_BCD_code = 0;
	write_outputs ();
	var_ST_flag = 0;
	var_CLK_flag = 0;
	var_current_state = Idle; 
	var_current_state_counter = Num_0; 
	
	INTERRUPT_PeripheralInterruptEnable();
		// Enable the Global Interrupts 
	INTERRUPT_GlobalInterruptEnable();

	// Do not use the TMR0 when Idle. 
	Counter_CLK_Stop();	
	
	
		// I2C client device port pins configuration as outputs
	__delay_ms(20);
	uint8_t data[MCP23008_DATALENGTH];
	
		//Configure GPIO as output
	data[0] = MCP23008_REG_ADDR_IODIR;
	data[1] = MCP23008_PINS_DIG_OUT;
	
	if (I2C->Write(MCP23008_I2C_CLIENT_ADDR, data, MCP23008_DATALENGTH)){
		while(I2C->IsBusy()){
			I2C->Tasks();
		}
		if (I2C->ErrorGet() == I2C_ERROR_NONE){
			/* Write operation is successful */
		}
		else {
			/* Error handling */
		}
	}
		// Initial write LED test: all LED high
	data[0] = MCP23008_REG_ADDR_GPIO;
	data[1] = MCP23008_PINS_DIG_HIGH;
	
		// Delay 0.5 s
	__delay_ms(500);
	if (I2C->Write(MCP23008_I2C_CLIENT_ADDR, data, MCP23008_DATALENGTH)){
		while(I2C->IsBusy()){
			I2C->Tasks();
		}
		if (I2C->ErrorGet() == I2C_ERROR_NONE){
			/* Write operation is successful */
		}
		else {
			/* Error handling */
		}
	}
	
		// All LED low: toggle bitwise operation
	data[1] = ~data[1];
		// Delay 0.5 s
	__delay_ms(1000);
	if (I2C->Write(MCP23008_I2C_CLIENT_ADDR, data, MCP23008_DATALENGTH)){
		while(I2C->IsBusy()){
			I2C->Tasks();
		}
		if (I2C->ErrorGet() == I2C_ERROR_NONE){
			/* Write operation is successful */
		}
		else {
			/* Error handling */
		}
	}
	
			// Delay 250 ms
	__delay_ms(250);	 
		 
	// --------------------------------------------------
	// The main control flow mechanist is based on executing the control FSM
	while(1){		// the FSM running in the infinite loop

		if (Cntl_FSM()) break;   

	}
	while (1){
		__nop();	// Assembler instruction meaning "do nothing"
					// It simply wastes 4 OSC cycles  
	/* The system has crashed in an illegal state and needs attention.
	This second infinite loop is never reached in normal operation. If by any 
	chance the system enters this loop, the only way to scape it is by
	clicking reset MCLR_L button (asyn CD_L) or provoking a software reset*/
	} 
}



/* 	===========================================================================
	FSM function 
		// State variable logic The combinational circuit CC1.
		// State transitions and loops: arrows in the state diagram 

		// Output logic: The combinational circuit CC2. 
		// Outputs in brackets in the state diagram. 
  	======================================================================== */
uint8_t Cntl_FSM(void){
	uint8_t error = 0;	

	switch (var_current_state) {
		case Idle:
		/*----------------------  CC1 state logic  ----------------------*/
			if (var_ST_flag == 1){
				var_current_state = Start_CLK;
				var_ST_flag =0;
			}
			else{
				var_current_state = Idle;  
			}
		/*----------------------  CC2 output logic  ---------------------*/
			var_BCD_code = 0;
			var_TC10 = 0;
			break;

		case Start_CLK:
		/*----------------------  CC1 state logic  ----------------------*/
			var_current_state = Counting;

		/*----------------------  CC2 output logic  ---------------------*/
		// Configure and enable the Counter CLK using MCC functions 
		// Set the blinking waveform TW/2
			Counter_CLK_PeriodSet(N2_5Hz);
		// Enable interrupts from TMR0 
			Counter_CLK_Start();
			
			break;

		case Counting:
		/*----------------------  CC1 state logic  ----------------------*/
			if (var_ST_flag == 1){
				var_current_state = Stop_CLK;
				var_ST_flag =0;
			}
			else{
				var_current_state = Counting;  
			}
		/*----------------------  CC2 output logic  ---------------------*/
			if (var_CLK_flag == 1){
				read_inputs(); // poll input switch value
				if (Counter_FSM()) 
					error = 1; 
				write_outputs();//The interface with the hardware to drive pins
				var_CLK_flag = 0;// Clear the var_CLK_flag after using it	
			}
			break;
			
		case Stop_CLK:
		/*----------------------  CC1 state logic  ----------------------*/
			var_current_state = Idle;
		
		/*----------------------  CC2 output logic  ---------------------*/
			// Disable the TMR0 using MCC functions
			Counter_CLK_Stop();
			var_BCD_code = 0;
			var_TC10 = 0;
		break;
		
		default:
		// If no states have been selected, there is an error
		// and the systems must be stopped
			error = 1;
		break;
	}
	return (error);
}


//*****************************************************************************
//Interrupt Service Routine for attending interrupts from INT0
//*****************************************************************************
void INT0_PB_CallBack(void){
	var_ST_flag = 1;
}
//*****************************************************************************
//Interrupt Service Routine for attending interrupts from TMR0
// Interrupt on TMR0 overflow, (every 250 ms)
//*****************************************************************************
static void TMR0_TP_CallBack(void){
	var_CLK_flag = 1;
}


/* ============================================================================
	Driving the output pins. Let's drive the output ports, converting 
	microcontroller variables into electrical signals.    
	As a rule in CSD we will write all the port pins in a single instruction
	preserving (re-writing) the pins of no interest 
	NOTE: LATx and PORTx registers are quite similar but not identical. 
	Let us use PORTx for reading data and LATx for writing data out
	======================================================================== */
void write_outputs(void){
 // These are convenient variables to save partial results while operating
	static uint8_t var_buff, var_buff2;
 
	//To write a given output pin:
	// Read and mask to preserve all the bits not to be written.
	// Clean all bits to be written
	// There are 2 LED at PORTC(3,2) used for displaying the code Q(3,2))
	var_buff = (PORTC & 0b11110011); 
	var_buff2 = var_buff | (var_BCD_code & 0b00001100); 
	LATC = var_buff2;		// Or PORTC = var_buff2; 

		// There are 2 LED at PORTB(5,4) used for displaying the code Q(1,0))
	var_buff = (PORTB & 0b11001111); 
	LATB = var_buff | ((var_BCD_code & 0b00000011) << 4);

		// TC10 is connected to a LED at PORTC pin 7 (RC7))
	var_buff = PORTC & 0b01111111; 
// Shift the bit of interest to the pin position, compose the port bits (OR)
	// and write the port in a single instruction
	var_buff2 = (var_TC10 << 7) | var_buff;
	LATC = var_buff2;
}


/* ============================================================================
 Scan, read, poll or capture inputs and process the bits so that we can set 
 our convenient variables for running the FSM.
 Many of the variables are "uint8_t" because we can easily monitor them using
 the watch window for debugging purposes.
  ========================================================================== */
void read_inputs (void){
// As we did in P9 (L9_3): 
// read the port, mask the bit of interest and save the variable 
	var_CE = (PORTA & 0b00010000)>>4;
}

/* ============================================================================
 Counter FSM implemented in software
  ========================================================================== */
uint8_t Counter_FSM(void){
	uint8_t error = 0;	

	switch (var_current_state_counter) {
		case Num_0:
		/*----------------------  CC1 state logic  ----------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_1;
		else
			var_current_state_counter = Num_0;  

		/*----------------------  CC2 output logic  ---------------------*/
		var_BCD_code = 0;
		var_TC10 = 0;
		break;

	case Num_1:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_2;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 1;
		var_TC10 = 0;
		break;

	case Num_2:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_3;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 2;
		var_TC10 = 0;
		break;

	case Num_3:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_4;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 3;
		var_TC10 = 0;
		break;

	case Num_4:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_5;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 4;
		var_TC10 = 0;
		break;

	case Num_5:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_6;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 5;
		var_TC10 = 0;
		break;

	case Num_6:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_7;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 6;
		var_TC10 = 0;
		break;

	case Num_7:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_8;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 7;
		var_TC10 = 0;
		break;

	case Num_8:
		/*----------------------  CC1 state logic  ------------------*/
		if (var_CE == 1)
			var_current_state_counter = Num_9;

	/*----------------------  CC2 output logic  -----------------*/
		var_BCD_code = 8;
		var_TC10 = 0;
		break;

	case Num_9:
		/*----------------------  CC1 state logic  -------------------*/
		if(var_CE == 1)
			var_current_state_counter = Num_0;
		else
			var_current_state_counter = Num_9;

		/*----------------------  CC2 output logic  ------------------*/
		var_BCD_code = 9;
		if (var_CE == 1)
			var_TC10 = 1;
		else
			var_TC10 = 0;
		break;

		default:
		// If no states have been selected, there is an error
		// and the systems must be stopped
			error = 1;
		break;
	}
	return (error);
}

/* end of file */