/* ============================================================================
 UPC - EETAC - CSD - http://digsys.upc.edu 

	Tutorials on learning and adapting MCC Melody

	Project: Lamp switch using the FSM control pattern and the A/D converter
	INT0 callback interrupt function connected to the ST/SP push-button
	LED bar output to display the binary value converted. 

	Compile with XC8 v3.1 compiler or newer (standard C99)
https://www.microchip.com/en-us/tools-resources/develop/mplab-xc-compilers/xc8
	Generate compilation output files COFF for Proteus simulations or ELF for 
	programming the prototype
	=========================================================================*/

/* ===============================================
	Prototype adapted to the CSD_PICstick training board
		https://digsys.upc.edu/csd/units/CSD_PICstick/CSD_PICstick.html
	Target microcontroller: PIC18F46K22 
		https://www.microchip.com/en-us/product/PIC18F46K22 

	ST_L push-button at RB0/INT0 
	Switch SW1 at RC4 is used to select connect the A/D
	Output LED bar Q(9..0) 

 Project documentation location:
	https://digsys.upc.edu/csd/units/MCC/mcc.html#T8

Microchip documentation:
www.microchip.com/en-us/tools-resources/configure/mplab-code-configurator
	======================================================================== */

/* On disabling some warnings: Disable warning 520 (function is never called)! 
To enable it just remove or comment the line below  */ 
#pragma warning disable 520
/* Other warnings can be disabled as compiler options: 
Properties --> XC8 Compiler options --> Additional options 
	-Xparser -Wno-implicit-int-float-conversion  -Wno-implicit-int-conversion  
*/
/* 	===========================================================================
	Headers
	======================================================================== */
// This general header "system.h" includes the "pic18f46k22.h" and all the 
// resources used in this application, automatically managed by MCC Melody.
#include "./FSM_AD_LEDbar_prj.X/mcc_generated_files/system/system.h"

/*	===========================================================================
	Definitions 
	======================================================================== */
// Our system has 2 states defined as follows:
#define Idle				'A'
#define Set_Acq_Timer		'B'
#define Wait_Sample_Pulse	'C'
#define Acquire_AN0			'D'
#define Write_Measurement	'E'
#define Stop_Acq_Timer		'F'

/* A RAM memory position (a single byte or 'char') "var_current_state" type 
"uint8_t" is used to encode state labels as ASCII capital letters for easy 
debugging. 
A single byte can encode up to 256 states, which is enough for our projects.
Remember that encoding FSM states was done using "sequential", "Gray", 
"one-hot", "Johnson", etc. in our previous Chapter 2 on VHDL and FPGA. */ 

/*	===========================================================================
	Function prototypes (not included in the MCC generated file structure
	======================================================================== */
static void INT0_PB_CallBack(void); // INT0 interrupt handler 
static void Sampling_CLK_CallBack(void); //TMR0 interrupt handler
void main(void); // main control flow

// The basic function to organise the code to emulate a FSM
// state logic and output logic in a switch case statement
uint8_t FSM(void); // return 0 when OK

// The auxiliary functions to read/poll and write port pins and be able to
// organise the software variables, to make the main program hardware
// independent and compatible for many microcontrollers. 
void write_outputs (void);
void read_inputs(void);  

/* 	===========================================================================
		Global variables and data types
Even if some variables are 'bit' type, we use 'uint8_t' to make them easy to 
watch while debugging.
============================================================================ */
uint8_t var_Lamp;	// at RB3 pin
uint8_t var_SF;	//SW1 value at RA4 pin 
uint16_t var_Q;	// The digital 10-bit AN0 value converted
				// Q(9..8) = RD(7..6); Q(7..2) = RC(7..2); Q(1..0) = RB(5..4)

//The flag that will be set in the interrupt service routine to indicate that 
// a new push-button edge is detected 
uint8_t var_ST_flag; 

//The flag that will be set in the interrupt service routine to indicate that 
// a new sampling period has elapsed 
uint8_t var_SP_flag; 

/* The state variable to be updated ("asynchronously") every time that 
"state_logic()" is executed. In this application we aim to synchronise the FSM 
with the external CLK as we did in Chapter 2. It is the FSM memory equivalent 
to the state register in Chapter 2	*/ 
static uint8_t var_current_state; 

// TODO: Replace TimerX with name of const struct TIMER_INTERFACE, 
// from MCC Generated Files > timer > tmrx.c  ----------
// This is the memory address where the timer callback is saved. 
static const struct TIMER_INTERFACE *Timer = &Sampling_CLK; 

/* 	===========================================================================
	main function 
	======================================================================== */
void main(void){
	
	SYSTEM_Initialize();// Automated and maintained by MCC Melody

	// Key elements to install interrupts: A pointer to our interrupt handler, 
	// in a way that we do not need to modify a single MCC generated file. 
	INT0_InterruptHandler = *INT0_PB_CallBack;
	
	// This structure element points to our TMR0 interrupt handler
	Timer->TimeoutCallbackRegister(Sampling_CLK_CallBack);
	
	// Initial values 
	var_Lamp = 0;
	
	var_ST_flag = 0;
	var_SP_flag = 0;
	write_outputs ();
	var_current_state = Idle; 
	var_Q = ADC_GetConversion(0);
	
	
	INTERRUPT_PeripheralInterruptEnable();
		// Enable the Global Interrupts 
	INTERRUPT_GlobalInterruptEnable();

	// Do not use the sampling period timer (TMR0) when Idle. 
	TMR0_Stop();
	
	while(1){		// the FSM running in the infinite loop
		read_inputs(); // poll input switch value
		if (FSM()) break;
		write_outputs(); //The interface with the hardware to convert variables
	}					// into voltages at a given microcontroller pin. 
	while (1){
		__nop();	// Assembler instruction meaning "do nothing"
					// It simply wastes 4 OSC cycles  
	/* The system has crashed in an illegal state and needs attention.
	This second infinite loop is never reached in normal operation. If by any 
	chance the system enters this loop, the only way to scape it is by
	clicking reset MCLR_L button (asyn CD_L) or provoking a software reset*/
	} 
}

/* 	===========================================================================
	FSM function 
		// State variable logic The combinational circuit CC1.
		// State transitions and loops: arrows in the state diagram 

		// Output logic: The combinational circuit CC2. 
		// Outputs in brackets in the state diagram. 
  	======================================================================== */
uint8_t FSM(void){
	uint8_t error = 0;	
	
		switch (var_current_state) {
			case Idle:
			/*----------------------  CC1 state logic  ----------------------*/
				if (var_ST_flag == 1){
					var_current_state = Set_Acq_Timer;
					var_ST_flag = 0;
				}
				else{
					var_current_state = Idle;  
				}
			/*----------------------  CC2 output logic  ---------------------*/
				var_Lamp = 0;

			break;

			case Set_Acq_Timer:
			/*----------------------  CC1 state logic  ----------------------*/
				var_current_state = Wait_Sample_Pulse;

			/*----------------------  CC2 output logic  ---------------------*/
				if (var_SF ==1) Sampling_CLK_PeriodSet(50000); //10 Hz
				else Sampling_CLK_PeriodSet(500); //1 kHz
				Sampling_CLK_Start();

				break;

			case Wait_Sample_Pulse:
/* This state can be replaced adding other features while waiting for samples*/
			/*----------------------  CC1 state logic  ----------------------*/
				if (var_SP_flag == 1){  // When TMRx overflows ...
				var_current_state = Acquire_AN0;
				var_SP_flag = 0;
				}
				else{
					var_current_state = Wait_Sample_Pulse;
				}

			/*----------------------  CC2 output logic  ---------------------*/
				var_Lamp = var_Lamp; // Do nothing;
			break;

			case Acquire_AN0:
			/*----------------------  CC1 state logic  ----------------------*/
				var_current_state = Write_Measurement;

			/*----------------------  CC2 output logic  ---------------------*/
				var_Q = ADC_GetConversion(0);
			break;

			case Write_Measurement:
/* This state can be replaced adding other features for processing data */
			/*----------------------  CC1 state logic  ----------------------*/
				if (var_ST_flag == 1){
				var_current_state = Stop_Acq_Timer;
				var_ST_flag = 0;
				}
				else{
					var_current_state = Wait_Sample_Pulse;
				}

			/*----------------------  CC2 output logic  ---------------------*/
// In this case, we simply toggle the Lamp value
				var_Lamp = (var_Lamp ^0x11111111)& 0b00000001;
			break;

			case Stop_Acq_Timer:
			/*----------------------  CC1 state logic  ----------------------*/
				var_current_state = Idle;

			/*----------------------  CC2 output logic  ---------------------*/
				var_Lamp = 0;
				Sampling_CLK_Stop();
				TMR0_Stop();
			break;

			default:
			// If no states have been selected, there is an error
			// and the systems must be stopped
				error = 1;	
			break;
		}
	return (error);
}

//*****************************************************************************
//Interrupt Service Routine for attending interrupts from INT0
//*****************************************************************************
static void INT0_PB_CallBack(void){
	var_ST_flag = 1;
}
//*****************************************************************************
//Interrupt Service Routine for attending the sampling CLK overflow
// You can choose six TMRx devices to fulfill this function. 
// MCC Melody abstracts this device management
//*****************************************************************************
static void Sampling_CLK_CallBack(void){
	var_SP_flag = 1;
}


/* ============================================================================
	Driving the output pins. Let's drive the output ports, converting 
	microcontroller variables into electrical signals.
	As a rule in CSD we will write all the port pins in a single instruction
	preserving (re-writing) the pins of no interest 
	NOTE: LATx and PORTx registers are quite similar but not identical. 
	Let us use PORTx for reading data and LATx for writing data out
   ========================================================================= */
void write_outputs(void){
// These are convenient local variables to save partial results while operating
	volatile uint8_t var_buff, var_buff2;
 
	//This is the way we write a given output pin:
		// Read and mask to preserve all the bits of no interest while 
		// cleaning all bits to be written.
	// var_Q occupies bits for three ports
	var_buff = (PORTB & 0b11001111); 
	var_buff2 = (var_Q & 0b0000000011)<< 4; 
	var_buff = var_buff |var_buff2; 
	LATB = var_buff; // write all the port pins in a single instruction 

	var_buff = (PORTC & 0b00000011); 
	var_buff2 = (var_Q & 0b0011111100); 
	var_buff = var_buff |var_buff2; 
	LATC = var_buff; // write all the port pins in a single instruction 

	var_buff = (PORTD & 0b00111111); 
	var_buff2 = (var_Q & 0b1100000000)>>2; 
	var_buff = var_buff |var_buff2; 
	LATD = var_buff; // write all the port pins in a single instruction 

	// There is a LED Lamp at PORTB3 used for displaying Lamp
	var_buff = (PORTB & 0b11110111); 
	var_buff2 = var_Lamp << 3; 
	var_buff = var_buff |var_buff2; 
	LATB = var_buff; // write all the port pins in a single instruction 
	// Note: the simulator generated weird results when this code for writing 
	// var_Lamp is the first to be written ?
}

/* ============================================================================
 Scan, read, poll or capture inputs and process the bits so that we can set 
 our convenient variables for running the FSM.
 Many of the variables are "uint8_t" because we can easily monitor them using
 the watch window for debugging purposes.
  ========================================================================== */
void read_inputs (void){
// This is the way we read or capture digital voltages:
	// (1) read the port
	// (2) mask the bit of interest
	// (3) shift to the bit0 position and save the convenient RAM variable 
	var_SF = (PORTA & 0b00010000)>>4;

}

/* end of file */