/* ============================================================================
 UPC - EETAC - CSD - http://digsys.upc.edu 

	Tutorials on learning and adapting MCC Melody

	Project design phase #1; step #2: Basic PWM using the FSM control pattern 
	INT0 callback interrupt function connected to the ST/SP push-button
	Lamp pilot ON/OFF output
	PWM_out output from CCP5 pin 

Compile with XC8 v3.1 compiler or newer (standard C99)
https://www.microchip.com/en-us/tools-resources/develop/mplab-xc-compilers/xc8
Generate compilation output files COFF for Proteus simulations or ELF for 
programming the prototype
=============================================================================*/

/* ===============================================
	Prototype adapted to the CSD_PICstick training board
		https://digsys.upc.edu/csd/units/CSD_PICstick/CSD_PICstick.html
	Target microcontroller: PIC18F46K22 
		https://www.microchip.com/en-us/product/PIC18F46K22 

	ST_L push-button at RB0/INT0 
	Switch PW at RC4 is used to read a single bit for selecting two DC values
	Output LED Lamp at RB5 

Project documentation location:
	https://digsys.upc.edu/csd/DEE/dimmer/dimmer.html#B2
	https://digsys.upc.edu/csd/DEE/dimmer/phaseB/PIC/Dimmer_PIC.html#top

Microchip documentation:
	https://onlinedocs.microchip.com/oxy/GUID-5A03F818-B7FC-4062-9792-57D08543B586-en-US-11/index.html
================================================== */

/* On disabling some warnings: Disable warning 520 (function is never called)! 
To enable it just remove or comment the line below  */
#pragma warning disable 520
/* Other warnings can be disabled as compiler options: 
Properties --> XC8 Compiler options --> Additional options 
	-Xparser -Wno-implicit-int-float-conversion  -Wno-implicit-int-conversion -Wno-#warnings
*/
/* ============================================================================
	Headers
============================================================================ */
// This general header "system.h" includes the "pic18f46k22.h" and all the 
// resources used in this application, automatically managed by MCC Melody.
#include "./PWM_prj.X/mcc_generated_files/system/system.h"

/* ============================================================================
	Definitions 
============================================================================ */
// Our system has 2 states defined as follows:
#define Idle			'A'
#define Start_PWM		'B'
#define Lamp_modulation	'C'
#define Stop_PWM		'D'

/* A RAM memory position (a single byte or 'char') "var_current_state" type 
"uint8_t" is used to encode state labels as ASCII capital letters for easy 
debugging. 
A single byte can encode up to 256 states, which is enough for our projects.
Remember that encoding FSM states was done using "sequential", "Gray", 
"one-hot", "Johnson", etc. in our previous Chapter 2 on VHDL and FPGA. */

/* ============================================================================
	Function prototypes (not included in the MCC generated file structure
============================================================================ */
static void INT0_PB_CallBack(void); // INT0 interrupt handler 
void main(void); // main control flow

// The basic function to organise the code to emulate a FSM
// state logic and output logic in a switch case statement
uint8_t FSM(void); // return 0 when OK

// The auxiliary functions to read/poll and write port pins and be able to
// organise the software variables, to make the main program hardware
// independent and compatible for many microcontrollers. 
void write_outputs(void);
void read_inputs(void);

/* ============================================================================
	Global variables and data types
Even if some variables are 'bit' type, we use 'uint8_t' to make them easy to 
watch while debugging.
============================================================================ */
uint8_t var_Lamp;
uint8_t var_PW; // To read the SW1
uint16_t var_DC; // Duty cycle (up to 1024 values) 
uint16_t var_DC_prev; // Previous duty cycle value 

//The flag that will be set in the interrupt service routine to indicate that 
// a new push-button edge is detected 
uint8_t var_ST_flag;

/* The state variable to be updated ("asynchronously") every time that 
"state_logic()" is executed. It is the FSM memory equivalent to the state 
register in Chapter 2	*/
static uint8_t var_current_state;

/* ============================================================================
	main function 
============================================================================ */
void main(void) {
	SYSTEM_Initialize(); // Automated and maintained by MCC Melody

	// Key elements to install interrupts: A pointer to our interrupt handler, 
	// in a way that we do not need to modify a single MCC generated file. 
	INT0_InterruptHandler = *INT0_PB_CallBack;

	// This structure element points to our TMR0 interrupt handler
	//Timer->TimeoutCallbackRegister(TMR0_TP_CallBack);

	// Initial values 
	var_Lamp = 0;

	var_ST_flag = 0;
	write_outputs();
	var_current_state = Idle;

	INTERRUPT_PeripheralInterruptEnable();
	// Enable the Global Interrupts 
	INTERRUPT_GlobalInterruptEnable();

	while (1) { // the FSM running in the infinite loop
		read_inputs(); // poll input switch value
		if (FSM()) break;
		write_outputs(); //The interface with the hardware to convert variables
	} // into voltages at a given microcontroller pin. 
	while (1) {
		__nop(); // Assembler instruction meaning "do nothing"
	// It simply wastes 4 OSC cycles  
		/* The system has crashed in an illegal state and needs attention.
	This second infinite loop is never reached in normal operation. If by any 
	chance the system enters this loop, the only way to scape it is by
	clicking reset MCLR_L button (asyn CD_L) or provoking a software reset*/
	}
}

/* 	===========================================================================
	FSM function 
	// State variable logic The combinational circuit CC1.
	// State transitions and loops: arrows in the state diagram 

	// Output logic: The combinational circuit CC2. 
	// Outputs in brackets in the state diagram. 
============================================================================ */
uint8_t FSM(void) {
	uint8_t error = 0;

	switch (var_current_state) {
		case Idle:
			/*----------------------  CC1 state logic  ----------------------*/
			if (var_ST_flag == 1) {
				var_current_state = Start_PWM;
				var_ST_flag = 0;
			} else var_current_state = Idle;

			/*----------------------  CC2 output logic  ---------------------*/
			var_Lamp = 0;
		break;

		case Start_PWM:
		/*----------------------  CC1 state logic  ----------------------*/
			var_current_state = Lamp_modulation;

		/*----------------------  CC2 output logic  ---------------------*/
			var_Lamp = 1;
			if (var_PW == 1) var_DC = 836;
			else var_DC = 353;
			var_DC_prev = var_DC; 
		// Set the PWM parameters 
			CCP5_Initialize(); //2 kHz waveform from MCC (WP = 1000)
			CCP5_LoadDutyValue(var_DC);
						//DC = 83.6% =>  var_PW = DC(var_WP)
						//35.3% =>  var_PW = 0.353(1000) = 353
							//83.6% =>  var_PW = 0.836(1000) = 836
			TMR2_Initialize();
		break;

		case Lamp_modulation:
			/*----------------------  CC1 state logic  ----------------------*/
			if (var_ST_flag == 1) {
				var_current_state = Stop_PWM;
				var_ST_flag = 0;
			} else var_current_state = Lamp_modulation;

			/*----------------------  CC2 output logic  ---------------------*/
			var_Lamp = 1;
			if (var_PW == 1) var_DC = 836;
			else var_DC = 353;
			if (var_DC != var_DC_prev){
				CCP5_LoadDutyValue(var_DC);
				var_DC_prev = var_DC;
			}
			
		break;

		case Stop_PWM:
		/*----------------------  CC1 state logic  ----------------------*/
			var_current_state = Idle;

		/*----------------------  CC2 output logic  ---------------------*/
		// Disable the peripheral PWM 
			var_Lamp = 0;
			//CCP5_Initialize(); 
			CCP5_LoadDutyValue(0);// Output CC5OUT (RE2 ) = 0
			TMR2_PeriodSet(0);
		break;
			
		default:
			// If no states have been selected, there is an error
			// and the systems must be stopped
			error = 1;
			break;
	}
	return (error);
}

/* ****************************************************************************
	//Interrupt Service Routine for attending interrupts from INT0
******************************************************************************/
void INT0_PB_CallBack(void) {
	var_ST_flag = 1;
}

/* ============================================================================
	Driving the output pins. Let's drive the output ports, converting 
microcontroller variables into electrical signals.
As a rule in CSD we will write all the port pins in a single instruction
preserving (re-writing) the pins of no interest 
NOTE: LATx and PORTx registers are quite similar but not identical. 
Let us use PORTx for reading data and LATx for writing data out
============================================================================ */
void write_outputs(void) {
// These are convenient local variables to save partial results while operating
	volatile uint8_t var_buff, var_buff2;

/* This is the way we write a given output pin:
	- Read and mask to preserve all the bits of no interest.
	- Clean all bits to be written.
	- Compose and write the port 

// There is one LED at PORTB(5) used for displaying the Lamp pilot */
	var_buff = (PORTB & 0b11011111);
	var_buff2 = var_Lamp << 5;
	var_buff = var_buff | var_buff2;
	LATB = var_buff; // write the port
}

/* ============================================================================
Scan, read, poll or capture inputs and process the bits so that we can set 
our convenient variables for running the FSM.
Many of the variables are "uint8_t" because we can easily monitor them using
the watch window for debugging purposes.
============================================================================ */
void read_inputs(void) {
	// As we did in P9 (L9_3): 
	// read the port, mask the bit of interest and save the variable 
	var_PW = (PORTA & 0b00010000)>>4;
}

/* end of file */