--------------------------------------------------------------------------------
-- Tutorial P7. Example of a 10-bit data register (10-bit word memory cell)
-- It is easily expandable to n-bit

-- Plans used: 
-- plan C1: three-block FSM architecture in a single file
-- plan Y: FSM with state signals defined as STD_LOGIC_VECTOR
-- Adapted from reference and schematics at:
-- https://digsys.upc.edu/csd/P07/Data_register/Data_Reg_4bit.html
-- UPC - EETAC - CSD 
-- http://digsys.upc.edu
--------------------------------------------------------------------------------
LIBRARY ieee;
USE  IEEE.STD_LOGIC_1164.all;

ENTITY Data_Reg_10bit IS
	Port (
		CLK	: IN	STD_LOGIC;
		CD	: IN	STD_LOGIC;
		LD	: IN	STD_LOGIC;
		Q	: OUT	STD_LOGIC_VECTOR(9 DOWNTO 0);
		Din	: IN 	STD_LOGIC_VECTOR(9 DOWNTO 0)
	);

END Data_Reg_10bit;

ARCHITECTURE FSM_like OF Data_Reg_10bit IS
-- Internal description in FSM style: 
-- Three blocks --> State register, CC1 and CC2

CONSTANT Reset	: STD_LOGIC_VECTOR(9 DOWNTO 0) := "0000000000"; 

-- Internal signals --> Plan Y:  
SIGNAL present_state,future_state: STD_LOGIC_VECTOR(9 DOWNTO 0);

BEGIN
--------------------------------------------------------------------------------
------- State register 
-- A set of D-type flip-flops in parallel
-- The asynchronous reset has precedence over the CLK's rising edge detection

State_Register: PROCESS (CD, CLK)
	BEGIN
						
	IF CD = '1' THEN -- An asynchronous reset called "Clear Direct"
		present_state <= Reset; 
	ELSIF (CLK'EVENT and CLK = '1') THEN -- Synchronous register (D_FF)
		present_state <= future_state; 
	END IF;

END PROCESS State_Register;

--------------------------------------------------------------------------------
-- CC1 ----- Combinational circuit for calculating next state to go
-- Even for simple FSM, this combinational circuit is better described always in 
-- the same way as a behavioural interpretation of the truth from its flow chart 
CC1: PROCESS (present_state, Din, LD)
	BEGIN
		IF LD = '1' THEN
			future_state <= Din;
		ELSE 	
			future_state <= present_state; -- keep the same state
		END IF;
END PROCESS CC1;

--------------------------------------------------------------------------------
-- CC2 ----- Combinational circuit for calculating the outputs
-- In this case,  we are simply interested in making available to an output 
-- the internal present_state

Q <= present_state; 

END FSM_like;



