--------------------------------------------------------------------------------
-- Synchronous universal 10-bit binary counter 
-- FSM style, plan Y: STD_LOGIC_VECTOR signals and IEEE.STD_LOGIC_ARITH library
-- This component is normally used as a building block for more complex circuits
---------------------------------------------------
-- Project P7  - CSD 
-- This code is an adaptation of the basic building block Counter_mod16 at:
-- https://digsys.upc.edu/csd/P07/Counter_mod16/Counter_mod16.html
--------------------------------------------------------------------------------

LIBRARY ieee;
USE  IEEE.STD_LOGIC_1164.all;
USE  IEEE.STD_LOGIC_ARITH.all;
USE  IEEE.STD_LOGIC_UNSIGNED.all;

ENTITY Counter_mod1024 IS
	Port (
		CLK		: IN	STD_LOGIC;
		CD		: IN	STD_LOGIC;
		Din		: IN 	STD_LOGIC_VECTOR(9 DOWNTO 0);
		LD		: IN	STD_LOGIC;
		CE		: IN	STD_LOGIC;
		UD_L	: IN	STD_LOGIC;
		Q		: OUT	STD_LOGIC_VECTOR(9 DOWNTO 0);
		TC1024	: OUT	STD_LOGIC
	);
END Counter_mod1024;

			-- Internal description in FSM style

ARCHITECTURE FSM_like OF Counter_mod1024 IS

CONSTANT Max_Count	: STD_LOGIC_VECTOR(9 DOWNTO 0):= "1111111111"; --1023
CONSTANT Reset		: STD_LOGIC_VECTOR(9 DOWNTO 0):= "0000000000"; 

-- Internal wires --> in this case just the present and future state signals

SIGNAL present_state,next_state: STD_LOGIC_VECTOR(9 DOWNTO 0);

BEGIN
--------------------------------------------------------------------------------
-- State register 
-- The only clocked block: 'r' data-type flip-flops (D_FF)
-- 'r' depends on the encoding of the FSM states. 
-- Asynchronous reset has precedence over the CLK rising edge
State_Register: PROCESS (CD, CLK, next_state)
	BEGIN
						
	IF CD = '1' THEN 	
		-- reset counter (asynchronous reset called "Clear Direct"
		present_state <= Reset; 
	ELSIF (CLK'EVENT and CLK = '1') THEN	
		-- Synchronous register (D_FF)
		present_state <= next_state; 
	END IF;

END PROCESS State_Register;

--------------------------------------------------------------------------------
-- CC1 Combinational circuit for calculating next state
--     Behavioural description of the truth table by means of a flow chart
CC1: PROCESS (present_state, LD, CE, UD_L, Din)
	BEGIN
	IF LD = '1' THEN 
		next_state <= Din; -- Load the parallel data (data register)

		ELSIF CE = '0' THEN
			next_state <= present_state; -- Do nothing -- Inhibit the counter

			ELSIF UD_L = '1' THEN
				IF(present_state < Max_Count ) THEN
					next_state <= present_state + 1 ; -- Up counter
				ELSE
					next_state <= Reset;
				END IF;	

			ELSE 	
				IF(present_state > Reset ) THEN
					next_state <= present_state - 1 ;  -- Down counter
				ELSE
					next_state <= Max_Count;
				END IF;	
						
	END IF;

END PROCESS CC1;
--------------------------------------------------------------------------------
-- CC_2: Combinational circuit for calculating the outputs

--Terminal count is a function with only 2 minterms:
TC1024 <= '1' WHEN CE = '1' and ((present_state = Max_count AND UD_L = '1') OR 
								(present_state = Reset AND UD_L = '0')) ELSE '0';

-- Data outputs are simply a buffer copy of the internal state: 
Q <= present_state; 

END FSM_like;





